﻿// Copyright (c) 2014 John Bledsoe
// Released under the MIT License
// https://github.com/jmbledsoe/angularjs-visualstudio-intellisense

(function (intellisense) {
    // If AngularJS is undefined, then bypass AngularJS Intellisense.
    if (!angular) {
        return;
    }

    // The following log levels are supported by AngularJS Intellisense.
    var LOG_LEVEL = {
        VERBOSE: 1,
        INFO: 2,
        WARN: 3,
        ERROR: 4,
        OFF: 5
    };

    // Set the current log level to one of the log levels above in order to view log messages.
    var CURRENT_LOG_LEVEL = LOG_LEVEL.OFF;

    // Expose a hidden object that may be used to set the log level from any file.
    window._$AngularJS_VisualStudio_Intellisense = {
        setLogLevelVerbose: function () { CURRENT_LOG_LEVEL = LOG_LEVEL.VERBOSE; },
        setLogLevelInfo: function () { CURRENT_LOG_LEVEL = LOG_LEVEL.INFO; },
        setLogLevelWarn: function () { CURRENT_LOG_LEVEL = LOG_LEVEL.WARN; },
        setLogLevelError: function () { CURRENT_LOG_LEVEL = LOG_LEVEL.ERROR; },
        setLogLevelOff: function () { CURRENT_LOG_LEVEL = LOG_LEVEL.OFF; }
    };

    //#region Logging Functions

    function indent(level) {
        var pad = '  '; // Two-space pad.

        return Array(level + 1).join(pad);
    }

    function logMessage(logLevel, message) {
        if (CURRENT_LOG_LEVEL > logLevel) {
            return;
        } else {
            intellisense.logMessage(message);
        }
    }

    function logValue(logLevel, value, key, level) {
        if (CURRENT_LOG_LEVEL > logLevel) {
            return;
        }

        if (angular.isUndefined(value)) {
            value = 'undefined';
        }

        if (angular.isUndefined(key)) {
            key = '';
        }

        if (angular.isUndefined(level)) {
            level = 0;
        }

        var pad = indent(level);

        if (angular.isString(value)) {
            value = '"' + value + '"';
        } else if (angular.isFunction(value)) {
            value = '$FUNCTION';
        } else if (angular.isElement(value)) {
            value = '$ELEMENT';
        } else if (isWindow(value)) {
            value = '$WINDOW';
        } else if (value && document === value) {
            value = '$DOCUMENT';
        } else if (isScope(value)) {
            value = '$SCOPE';
        }
        if (angular.isArray(value)) {
            intellisense.logMessage(pad + (key ? key + ': ' : '') + ' [');
            forEach(value, function (item) {
                logValue(logLevel, item, '', level + 1);
            });

            intellisense.logMessage(pad + ']' + (level > 0 ? ',' : ''));
        } else if (angular.isObject(value)) {
            if (filter(value)) {
                intellisense.logMessage(pad + (key ? key + ': ' : '') + '{');

                forEach(value, function (propertyValue, key) {
                    logValue(logLevel, propertyValue, key, level + 1);
                });

                intellisense.logMessage(pad + '}' + (level > 0 ? ',' : ''));
            }
        } else {
            intellisense.logMessage(pad + (key ? key + ': ' : '') + value + (level > 0 ? ',' : ''));
        }
    }

    //#endregion

    //#region Utility Functions

    function forEach(obj, iterator, context) {
        var key;
        if (obj.forEach && obj.forEach !== forEach) {
            obj.forEach(iterator, context);
        } else if (angular.isArray(obj)) {
            for (key = 0; key < obj.length; key++) {
                iterator.call(context, obj[key], key);
            }
        } else {
            for (key in obj) {
                iterator.call(context, obj[key], key);
            }
        }

        return obj;
    }

    // Copy AngularJS functions for determining window or $scope objects.
    function isWindow(obj) {
        return obj && obj.document && obj.location && obj.alert && obj.setInterval;
    }

    function isScope(obj) {
        return obj && obj.$evalAsync && obj.$watch;
    }

    //#endregion

    //#region $compile.directive.Attributes

    // HACK: (JMB)  Since the directive attributes type is hidden and "difficult" to access during Intellisense generation,
    //              (i.e. I couldn't figure it out), create a copy of it here so that Intellisense treats it correctly.
    var Attributes = function (element, attr) {
        this.$$element = element;
        this.$attr = attr || {};
    };

    Attributes.prototype = {
        $normalize: function (name) {
            return '';
        },


        /**
         * @ngdoc method
         * @name $compile.directive.Attributes#$addClass
         * @kind function
         *
         * @description
         * Adds the CSS class value specified by the classVal parameter to the element. If animations
         * are enabled then an animation will be triggered for the class addition.
         *
         * @param {string} classVal The className value that will be added to the element
         */
        $addClass: function (classVal) {
        },

        /**
         * @ngdoc method
         * @name $compile.directive.Attributes#$removeClass
         * @kind function
         *
         * @description
         * Removes the CSS class value specified by the classVal parameter from the element. If
         * animations are enabled then an animation will be triggered for the class removal.
         *
         * @param {string} classVal The className value that will be removed from the element
         */
        $removeClass: function (classVal) {
        },

        /**
         * @ngdoc method
         * @name $compile.directive.Attributes#$updateClass
         * @kind function
         *
         * @description
         * Adds and removes the appropriate CSS class values to the element based on the difference
         * between the new and old CSS class values (specified as newClasses and oldClasses).
         *
         * @param {string} newClasses The current CSS className value
         * @param {string} oldClasses The former CSS className value
         */
        $updateClass: function (newClasses, oldClasses) {
        },

        /**
         * Set a normalized attribute on the element in a way such that all directives
         * can share the attribute. This function properly handles boolean attributes.
         * @param {string} key Normalized key. (ie ngAttribute)
         * @param {string|boolean} value The value to set. If `null` attribute will be deleted.
         * @param {boolean=} writeAttr If false, does not write the value to DOM element attribute.
         *     Defaults to true.
         * @param {string=} attrName Optional none normalized name. Defaults to key.
         */
        $set: function (key, value, writeAttr, attrName) {
        },


        /**
         * @ngdoc method
         * @name $compile.directive.Attributes#$observe
         * @kind function
         *
         * @description
         * Observes an interpolated attribute.
         *
         * The observer function will be invoked once during the next `$digest` following
         * compilation. The observer is then invoked whenever the interpolated value
         * changes.
         *
         * @param {string} key Normalized key. (ie ngAttribute) .
         * @param {function(interpolatedValue)} fn Function that will be called whenever
                  the interpolated value of the attribute changes.
         *        See the {@link guide/directive#Attributes Directives} guide for more info.
         * @returns {function()} the `fn` parameter.
         */
        $observe: function (key, fn) {
            return fn;
        }
    };

    //#endregion

    //#region Module Tracking

    // Keep track of module names, with each module name mapped to an array of all its required modules.
    var requiredModuleMap = {};
    var moduleProviderFunctions = ['provider', 'factory', 'service', 'animation', 'filter', 'controller', 'directive', 'config', 'run'];

    // Keep track of the provider injector in order to inject into providers.
    var providerInjector;

    angular.module('ng').config(['$injector', '$provide', function ($injector, $provide) {
        // Keep track of the the provider injector.
        providerInjector = $injector;

        $provide.decorator("$http", ['$delegate', function ($delegate) {
            /**
            * @typedef {Object} httpConfig
            * @property {String} method HTTP method (e.g. 'GET', 'POST', etc)
            *    @property {String} url Absolute or relative URL of the resource that is being requested.
            * @property {Object.<string|Object>} params  – Map of strings or objects which will be turned
            *      to `?key1=value1&key2=value2` after the url. If the value is not a string, it will be
            *      JSONified.
            * @property {string|Object} data  – Data to be sent as the request message data.
            * @property {Object} headers  – Map of strings or functions which return strings representing
            *      HTTP headers to send to the server. If the return value of a function is null, the
            *      header will not be sent.
            * @property {string} xsrfHeaderName  – Name of HTTP header to populate with the XSRF token.
            * @property {string} xsrfCookieName  – Name of cookie containing the XSRF token.
            * @property {function(data, headersGetter)|Array.<function(data, headersGetter)>} transformRequest  –
            *      transform function or an array of such functions. The transform function takes the http
            *      request body and headers and returns its transformed (typically serialized) version.
            * @property {function(data, headersGetter)|Array.<function(data, headersGetter)>} transformResponse  –
            *      transform function or an array of such functions. The transform function takes the http
            *      response body and headers and returns its transformed (typically deserialized) version.
            * @property {boolean|Cache} cache  – If true, a default $http cache will be used to cache the
            *      GET request, otherwise if a cache instance built with
            *      {@link ng.$cacheFactory $cacheFactory}, this cache will be used for
            *      caching.
            * @property {number|Promise} timeout  – timeout in milliseconds, or {@link ng.$q promise}
            *      that should abort the request when resolved.
            * @property {boolean} withCredentials  - whether to set the `withCredentials` flag on the
            *      XHR object.
            * @property {string} responseType  
            */

            /**
            * @param {string} url
            * @param {httpConfig} [config]
            */
            function httpGetDocs(url, config) {
            }
            /**
            * @param {string} url
            * @param {httpConfig} [config]
            */
            function httpHeadDocs(url, config) { }
            /**
            * @param {string} url
            * @param {httpConfig} [config]
            */
            function httpJsonpDocs(url, config) { }
            /**
            * @param {string} url
            * @param {httpConfig} [config]
            */
            function httpDeleteDocs(url, config) { }
            /**
            * @param {string} url
            * @param {*} data
            * @param {httpConfig} [config]
            */
            function httpPostDocs(url, data, config) { }
            /**
            * @param {string} url
            * @param {*} data
            * @param {httpConfig} [config]
            */
            function httpPutDocs(url, data, config) {
            }
            /**
            * @param {string} url
            * @param {*} data
            * @param {httpConfig} [config]
            */
            function httpPatchDocs(url, data, config) {
            }

            intellisense.annotate($delegate.get, httpGetDocs);
            intellisense.annotate($delegate.delete, httpDeleteDocs);
            intellisense.annotate($delegate.jsonp, httpJsonpDocs);
            intellisense.annotate($delegate.head, httpHeadDocs);
            intellisense.annotate($delegate.post, httpPostDocs);
            intellisense.annotate($delegate.put, httpPutDocs);
            intellisense.annotate($delegate.patch, httpPatchDocs);
            return $delegate;
        }]);

        // Decorate the $q service to resolve deferred objects at the end of the digest cycle.
        $provide.decorator('$q', ['$rootScope', '$delegate', function ($rootScope, $delegate) {
            var originalDefer = $delegate.defer;

            $delegate.defer = function () {
                // Create a deferred object.
                var deferred = originalDefer.apply($delegate, arguments);
                var promise = deferred.promise;

                // Override the promise methods to call handlers after the digest cycle.
                // This allows them to be called with parameters by user code, but if they
                // are never called on the first digest cycle they will still get Intellisense
                // on closure variables.
                function callArgsAfterDigest(originalFunc) {
                    return function () {
                        forEach(arguments, function (argument) {
                            if (angular.isFunction(argument)) {
                                $rootScope.$$postDigest(argument);
                            }
                        });

                        return originalFunc.apply(promise, arguments);
                    };
                }

                var originalThen = promise.then,
                    originalCatch = promise['catch'],
                    originalFinally = promise['finally'];

                promise.then = callArgsAfterDigest(originalThen);
                promise['catch'] = callArgsAfterDigest(originalCatch);
                promise['finally'] = callArgsAfterDigest(originalFinally);

                return deferred;
            };

            return $delegate;
        }]);

        // Decorate the $httpBackend service to execute the callback rather than using 
        // XHR, so that functions handling the response are called during Intellisense.
        $provide.decorator('$httpBackend', [function () {
            return function (method, url, post, callback) {
                callback(200, undefined, '', 'OK');
            };
        }]);

        // Decorate the $rootScope to always call event listeners registered 
        // with $on, so that listener functions are called during Intellisense.
        $provide.decorator('$rootScope', ['$delegate', function ($delegate) {
            var original$On = $delegate.$on;

            $delegate.$on = function (name) {
                $delegate.$$postDigest(function () {
                    $delegate.$emit(name);
                });

                return original$On.apply($delegate, arguments);
            };

            return $delegate;
        }]);
    }]);

    // Decorate angular.forEach to always call the callback once, even if it wouldn't 
    // normally be called, so that closure variables will be available via Intellisense.
    var originalForEach = angular.forEach;

    angular.forEach = function (obj, iterator, context) {
        var iteratorCalled = false;
        originalForEach.call(angular, obj, function () {
            iteratorCalled = true;
            iterator.apply(context, arguments);
        }, context);

        if (!iteratorCalled) {
            iterator.call(context, undefined, '');
        }

        return obj;
    };

    intellisense.redirectDefinition(angular.forEach, originalForEach);

    function isAngularModule(obj) {
        // Angular modules all have names and invoke queues and core provider functions.
        return angular.isObject(obj) &&
            angular.isString(obj.name) &&
            angular.isArray(obj._invokeQueue) &&
            angular.isFunction(obj.provider) &&
            angular.isFunction(obj.constant) &&
            angular.isFunction(obj.value) &&
            angular.isFunction(obj.factory) &&
            angular.isFunction(obj.service);
    }

    // Decorate the angular.module function to record the name of each module as it is registered.
    var originalModuleFunction = angular.module;

    angular.module = function (name, requires, configFn) {
        var hasModuleBeenTracked = requiredModuleMap[name] !== undefined;

        logMessage(LOG_LEVEL.VERBOSE, 'Calling "angular.module" with the following arguments:');
        logValue(LOG_LEVEL.VERBOSE, arguments);

        // If the module has not yet been tracked, then call the original module function with all of the specified arguments.
        // Otherwise, call the original module function with only the module name.
        // (This prevents the module from being recreated if the module is being declared in this file but is referred to in other files.)
        var returnValue = hasModuleBeenTracked ?
            originalModuleFunction.call(angular, name) :
            originalModuleFunction.apply(angular, arguments);

        // When editing a file that initially creates a module (e.g. via angular.module('name', ['dependency']))
        // it's likely that the JS editor has already executed code that uses that module and did not define dependencies
        // (e.g. via angular.module('name')). This line of code makes sure that the dependencies are maintained.
        if (!returnValue.requires && requires) {
            returnValue.requires = requires;
        }

        // HACK: For some reason, the implicit require of the 'ng' module gets dropped in the call to the originalModuleFunction
        // this re-adds it if it wasn't explicit
        if (angular.isArray(returnValue.requires) && requires.indexOf('ng') == -1) {
            returnValue.requires = requires.concat('ng');
        }

        // Ensure that the module and its dependencies are tracked, and all of its provider functions run.
        trackModule(returnValue);

        // Call the configuration function if one is specified.
        if (configFn) {
            returnValue.config(configFn);
        }

        return returnValue;
    };
    intellisense.redirectDefinition(angular.module, originalModuleFunction);

    function trackModule(moduleOrName) {
        var moduleName, module;

        // Tell the JavaScript editor that progress is being made in building the
        // IntelliSense simulation, giving us more time to process modules before timing out
        intellisense.progress();

        if (angular.isString(moduleOrName)) {
            // If the argument is a module name, retrieve the module from the angular.module function.
            moduleName = moduleOrName;
            module = originalModuleFunction.call(angular, moduleName);
        } else {
            // Otherwise the argument is a module, so get the name from its name property.
            module = moduleOrName;
            moduleName = module.name;
        }

        if (requiredModuleMap[moduleName] === undefined) {
            logMessage(LOG_LEVEL.INFO, 'Tracking module "' + moduleName + '".');

            // Store the module name mapped to the names of all required modules.
            var requiredModuleNames = [moduleName];

            // Recursively process dependent modules.
            forEach(module.requires, function (requiredModuleName) {
                trackModule(requiredModuleName);
                requiredModuleNames.splice(requiredModuleNames.length, 0, requiredModuleMap[requiredModuleName]);
            });

            requiredModuleMap[moduleName] = requiredModuleNames;

            // Decorate module provider functions.
            decorateModuleProviderFunctions(module);
        }
    }

    function decorateModuleProviderFunctions(module) {
        function addNavBarOverride(name, providerFn, callBackDefinition) {
            if (!intellisense.declareNavigationContainer) {
                return;
            }

            // When the callback defintion is an array, pull the actual callback off the end
            if (angular.isArray(callBackDefinition)) {
                callBackDefinition = callBackDefinition[callBackDefinition.length - 1];
            }

            // Add an entry to the nav bar for the current provider function
            intellisense.declareNavigationContainer(
            { callback: callBackDefinition },
                name + ' (' + providerFn + ')',
                'vs:GlyphGroupType');
        }

        // Initialize each component with empty object dependencies. 
        forEach(moduleProviderFunctions, function (providerFunction) {

            // Decorate the component type function to call component functions with correct arguments.
            var originalProviderFunction = module[providerFunction];

            // Only decorate the provider function if the module has it (which it may not for animate).
            if (originalProviderFunction) {
                module[providerFunction] = function (name, callBackDefinition) {
                    logMessage(LOG_LEVEL.VERBOSE, 'Calling provider function "' + providerFunction + '" with the following arguments:');
                    logValue(LOG_LEVEL.VERBOSE, arguments);

                    // Call the original component type function.
                    var returnValue = originalProviderFunction.apply(module, arguments);

                    // Create an injector for the module.
                    // (This will execute all configuration and run blocks.)
                    var injector = angular.injector(requiredModuleMap[module.name]);

                    if (arguments.length === 2) {
                        var component;
                        var locals;

                        // Factories, services, providers, etc.
                        logMessage(LOG_LEVEL.INFO, 'Creating instance of ' + providerFunction + ' "' + arguments[0] + '".');
                        addNavBarOverride(name, providerFunction, callBackDefinition);

                        // Before calling the injector, make sure the JS editor knows that progress has been made.
                        // This helps avoid a "timeout" situation.
                        intellisense.progress();

                        // Initialize the component based on the provider function.
                        switch (providerFunction) {
                            case 'factory':
                            case 'service':
                                component = injector.get(arguments[0]);

                                break;
                            case 'provider':
                                var component = arguments[1];

                                if (angular.isArray(component) || angular.isFunction(component)) {
                                    component = providerInjector.instantiate(component);
                                }

                                break;
                            case 'controller':
                                // Create locals to aid with injection.
                                locals = {
                                    '$scope': injector.get('$rootScope').$new()
                                };

                                component = injector.get('$controller')(arguments[0], locals);

                                break;
                            case 'filter':
                                component = injector.get('$filter')(arguments[0]);

                                break;
                            case 'directive':
                            case 'animation':
                                // Create an instance of the directive/animation definition object.
                                component = injector.invoke(arguments[1]);

                                break;
                        }

                        logMessage(LOG_LEVEL.VERBOSE, 'Creating instance of ' + providerFunction + ' "' + arguments[0] + '" returned the following:');
                        logValue(LOG_LEVEL.VERBOSE, component);

                        if (providerFunction === 'directive') {
                            // HACK: (JMB) Execute directive functions with AngularJS mocks.
                            var controller,
                                $scope = injector.get('$rootScope').$new(),
                                element = angular.element(),
                                attrs = new Attributes(),
                                transclude = function (scope, cloneLinkFn) { };

                            if (component.controller) {
                                logMessage(LOG_LEVEL.INFO, 'Calling function "controller" on directive definition object.');

                                controller = injector.instantiate(component.controller, {
                                    '$scope': $scope,
                                    '$element': element,
                                    '$attrs': attrs,
                                    '$transclude': transclude
                                });

                                logMessage(LOG_LEVEL.VERBOSE, 'Calling function "controller" on directive definition object returned the following:');
                                logValue(LOG_LEVEL.VERBOSE, controller);
                            }

                            if (component.compile) {
                                logMessage(LOG_LEVEL.INFO, 'Calling function "compile" on directive definition object.');

                                // Set the result of the compile function as the directive link function, so it is called as well.
                                component.link = component.compile(element, attrs, transclude);

                                logMessage(LOG_LEVEL.VERBOSE, 'Calling function "compile" on directive definition object returned the following:');
                                logValue(LOG_LEVEL.VERBOSE, component.link);
                            }

                            if (component.link) {
                                if (angular.isFunction(component.link)) {
                                    logMessage(LOG_LEVEL.INFO, 'Calling function "link" on directive definition object.');

                                    var returnValue = component.link($scope, element, attrs, controller, transclude);

                                    logMessage(LOG_LEVEL.VERBOSE, 'Calling function "link" on directive definition object returned the following:');
                                    logValue(LOG_LEVEL.VERBOSE, returnValue);
                                }
                                if (component.link.pre) {
                                    logMessage(LOG_LEVEL.INFO, 'Calling function "link.pre" on directive definition object.');

                                    var returnValue = component.link.pre($scope, element, attrs, controller, transclude);

                                    logMessage(LOG_LEVEL.VERBOSE, 'Calling function "link.pre" on directive definition object returned the following:');
                                    logValue(LOG_LEVEL.VERBOSE, returnValue);
                                }
                                if (component.link.post) {
                                    logMessage(LOG_LEVEL.INFO, 'Calling function "link.post" on directive definition object.');

                                    var returnValue = component.link.post($scope, element, attrs, controller, transclude);

                                    logMessage(LOG_LEVEL.VERBOSE, 'Calling function "link.post" on directive definition object returned the following:');
                                    logValue(LOG_LEVEL.VERBOSE, returnValue);
                                }
                            }
                        } else if (providerFunction === 'animation') {
                            // HACK: (JMB) Execute animate functions with AngularJS mocks.
                            var element = angular.element(),
                            doneCallback = function () { };

                            forEach(component, function (value, key) {
                                if (angular.isFunction(value)) {
                                    var returnValue;
                                    logMessage(LOG_LEVEL.INFO, 'Calling function "' + key + '" on animation definition object.');

                                    switch (key) {
                                        case 'enter':
                                        case 'leave':
                                        case 'move':
                                            returnValue = value.call(component, element, doneCallback);
                                            break;
                                        case 'addClass':
                                        case 'removeClass':
                                            returnValue = value.call(component, element, '', doneCallback);
                                            break;
                                        default:
                                            returnValue = value.call(component);
                                            break;
                                    }

                                    logMessage(LOG_LEVEL.VERBOSE, 'Calling function "' + key + '" on animation definition object returned the following:');
                                    logValue(LOG_LEVEL.VERBOSE, returnValue);
                                }
                            });
                        } else {
                            // Execute all functions on the initialized component.
                            callComponentFunctions(injector, component, locals);
                        }

                        // Digest the root scope to force promise resolution.
                        injector.get('$rootScope').$digest();

                        return returnValue;
                    } else {
                        // In all other cases, force the provider function to behave the same as normal
                        // This was required for .config() functions to work correctly.
                        return returnValue;
                    }
                };
            }
        });
    }
    function callComponentFunctions(injector, component, locals, recursionDepth) {
        // A recursion guard, to prevent this code from recursing too long and
        // causing the IntelliSense engine to timeout
        if (!recursionDepth) {
            recursionDepth = 0;
        }
        if (recursionDepth++ >= 2) {
            return;
        }

        // Tell the JavaScript editor that progress is being made in building the
        // IntelliSense simulation, giving us more time to call component functions before timing out
        intellisense.progress();

        if (component) {
            if (angular.isElement(component) || angular.isString(component) || angular.isNumber(component) || angular.isDate(component)) {
                // Bypass calling component functions when there likely aren't any user-defined
                // functions to call
                return;
            } else if (angular.isArray(component) || angular.isFunction(component)) {
                // If the component itself is a function, then call it.
                logMessage(LOG_LEVEL.INFO, 'Calling component as function.');
                var returnValue = injector.invoke(component, null, locals);
                logMessage(LOG_LEVEL.VERBOSE, 'Calling component as function returned the following:');
                logValue(LOG_LEVEL.VERBOSE, returnValue);

                // Recursively call functions on the return value.
                callComponentFunctions(injector, returnValue, locals, recursionDepth);
            } else {
                logMessage(LOG_LEVEL.VERBOSE, 'Calling all functions on the following component:');
                logValue(LOG_LEVEL.VERBOSE, component);

                // Call each function that is a property of the component.
                forEach(component, function (value, key) {
                    if (angular.isArray(value) || angular.isFunction(value)) {
                        logMessage(LOG_LEVEL.INFO, 'Calling function "' + key + '" on component.');
                        var returnValue = injector.invoke(value, component, locals);
                        logMessage(LOG_LEVEL.VERBOSE, 'Calling function "' + key + '" on component returned the following:');
                        logValue(LOG_LEVEL.VERBOSE, returnValue);

                        // Recursively call functions on the return value.
                        callComponentFunctions(injector, returnValue, locals, recursionDepth);
                    }
                });
            }
        }
    }

    // Always add the AngularJS core module.
    trackModule('ng');

    //#endregion

    //#region Jasmine Intellisense

    if (jasmine) {
        // Create an array of functions to override.
        var overrides = [
            'describe', 'xdescribe',
            'beforeEach', 'afterEach',
            'it', 'xit',
            'expect',
            'module', 'inject',
            {source: angular.mock, method: 'module'},
            {source: angular.mock, method: 'inject'}
        ];

        var jasmineInjector;

        forEach(overrides, function (override) {
            // Extract source and method from the override.
            var source = override.source || window;
            var method = override.method || override;

            source[method] = function () {
                // Don't actually call the original method, since it interferes with Intellisense.

                if (method == 'module') {
                    // Track each named module, call each anonymous module.
                    forEach(arguments, function (argument) {
                        if (angular.isString(argument)) {
                            // Track the module.
                            trackModule(argument);

                            // (Re)create an injector for the module.
                            jasmineInjector = angular.injector(requiredModuleMap[argument]);
                        } else if (angular.isFunction(argument) || angular.isArray(argument)) {
                            // Invoke the module configuration function.
                            providerInjector.invoke(argument);
                        }
                    });
                } else if (method == 'inject') {
                    // Perform injection on each argument of the method.
                    forEach(arguments, function (argument) {
                        jasmineInjector.invoke(argument);
                    });
                } else {
                    // Otherwise, call any function arguments to the method.
                    forEach(arguments.filter(angular.isFunction), function (argument) {
                        argument();
                    });

                    if (method === 'expect') {
                        // Return an expectation from calls to expect.
                        return jasmine.Expectation.Factory();
                    }
                }
            };
        });
    }

    //#endregion

    // Filter out private AngularJS properties (prefixed with $$) from statement completion.
    if (intellisense && intellisense.addEventListener) {
        intellisense.addEventListener('statementcompletion', function (event) {
            var filterRegex = /^\$\$.*/;

            event.items = event.items.filter(function (item) {
                return !filterRegex.test(item.name);
            });
        });
    }
})(window.intellisense);
// SIG // Begin signature block
// SIG // MIIkEwYJKoZIhvcNAQcCoIIkBDCCJAACAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // 4krYDaHJWF+vb0UyvHnCKCLqgw8EGzhwEjQgKSAPBKGg
// SIG // gg2TMIIGETCCA/mgAwIBAgITMwAAAI6HkaRXGl/KPgAA
// SIG // AAAAjjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTE2MTExNzIyMDkyMVoX
// SIG // DTE4MDIxNzIyMDkyMVowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBANCH1EIrfp3ZxnrUosPjFZLS
// SIG // U52VF8lSNvpUv6sQr+nJ58wmU8PCc79t1gDlANzpamc0
// SIG // MPYWF7QBpZV8i7mkLOaLO3n2Iwx5j/NS30ABHMLGA53r
// SIG // Wc9z6dhxOZvwziVZLdLJWwrvftYyDl10EgTsngRTpmsC
// SIG // Z/hNWYt34Csh4O/ApEUSzwN7A8Y5w9Qi3FVcd0L/nLLl
// SIG // VWdoui12an9mU0fVRwrMON6Ne5cZfYLQJviljuWh8F5k
// SIG // EOT56yfG8uAI0A3yZ8DY8i/7idoV+a4PPgCXB9ELPnDU
// SIG // d6tyeEGYB7gXzKKxX+y981Bno9eU8NKLVY9TppWT5rJm
// SIG // z8k3aORjx88CAwEAAaOCAYAwggF8MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjdMCAEGCCsGAQUFBwMDMB0GA1UdDgQWBBSr
// SIG // yNbtshXSqo7xzO1sOPdFStCKuzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwMDEy
// SIG // K2IwNTBjNmU3LTc2NDEtNDQxZi1iYzRhLTQzNDgxZTQx
// SIG // NWQwODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzcitW2o
// SIG // ynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEGCCsGAQUF
// SIG // BwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwDQYJKoZIhvcNAQELBQADggIBAESJAqxpU/PE
// SIG // trvUjGBT58psqElpZr6lmkGZOtid0lcCUWr6v5uW26Ym
// SIG // fQlW6NztJXV6pUdSqB5LFlPz7g+awwSVKcGChKRWMfyg
// SIG // ipGVtb9azqkBH2RGoebK8dd0e7+SCFFefDMCXlE7m+XY
// SIG // Ll8CTAmcGkPace3k2eei2nQsF63lDLUY9VQJ1L4cc80g
// SIG // e6T6yNvY2zqu+pDFo72VZa5GLVcpWNaS8GzaY/GPM6J+
// SIG // OHZe3fM17ayaO2KB0E4ZfEh8sAuPOMwtvNU5ZamVwQPi
// SIG // ksm5q9JXCqrcUgsuViej4piXV468qVluJJKOguIJc4LZ
// SIG // NYPMn3/RBI6IuOKag1iw1JrmMfqUR459puJOefPY02oz
// SIG // FlBw8UK7mAnp/8yVVVsIv5JSqAjE8ejx/0DX+Zo2nf26
// SIG // kIXSVT5QrUYf7yUMuJ46SARj73iYol0DDQLY3CCr5la1
// SIG // 3u8WZsPXVYIeT4J4yZ5UGhBgtxerQBORrrAZwZozne4y
// SIG // cs1lzE9GmC0PUWAefPv+2+gHeQf3oTM4/gma2497tjq9
// SIG // hYa4zLx9ATC3ex2pXRu9zE0X925HM9VA32rKLlG4tbnP
// SIG // wwTTO+Xj6RCM66e63qQuM2opLxRK6h7BIjg1BYXvwgQA
// SIG // DWvB2JYUSBWvflKwuGDEUrVKgreFKgBJKiaDJ1pB3r3V
// SIG // Zkm8C5x4cAm8MIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAA
// SIG // AzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2Vy
// SIG // dGlmaWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4
// SIG // MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29k
// SIG // ZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00
// SIG // uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03a8YS2Avw
// SIG // OMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+b
// SIG // U7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQ
// SIG // z7NEt13YxC4Ddato88tt8zpcoRb0RrrgOGSsbmQ1eKag
// SIG // Yw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAf
// SIG // TVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+E
// SIG // GvKhL1nkkDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVO
// SIG // VpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSH
// SIG // vMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rh
// SIG // KEmdX4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3
// SIG // s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
// SIG // w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
// SIG // sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ec
// SIG // XL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90lfdu
// SIG // +HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
// SIG // jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2oynUClTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBRyLToCMZBDuRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBR
// SIG // ME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDEx
// SIG // XzIwMTFfMDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBO
// SIG // BggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIw
// SIG // MTFfMDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsG
// SIG // AQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
// SIG // cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
// SIG // mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74
// SIG // w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11lhJB9i0ZQ
// SIG // VdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeU
// SIG // OeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb
// SIG // 7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0wI/zRive/DvQ
// SIG // vTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLw
// SIG // xS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn
// SIG // +N4sOiBpmLJZiWhub6e3dMNABQamASooPoI/E01mC8Cz
// SIG // TfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jy
// SIG // FqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw
// SIG // 3MYbBL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtI
// SIG // EJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
// SIG // 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
// SIG // gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF6
// SIG // 70EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr/Xmf
// SIG // wb1tbWrJUnMTDXpQzTGCFdgwghXUAgEBMIGVMH4xCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29m
// SIG // dCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAACOh5Gk
// SIG // Vxpfyj4AAAAAAI4wDQYJYIZIAWUDBAIBBQCggcYwGQYJ
// SIG // KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
// SIG // NwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkE
// SIG // MSIEIJVNBjcrACbYjFEj3CTZakZMC7Ow8LS64zzGJfzg
// SIG // veUBMFoGCisGAQQBgjcCAQwxTDBKoDCALgBhAG4AZwB1
// SIG // AGwAYQByAC4AaQBuAHQAZQBsAGwAaQBzAGUAbgBzAGUA
// SIG // LgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5jb20wDQYJ
// SIG // KoZIhvcNAQEBBQAEggEAAQtxBzSzE50RQBSKc+Vr+GeM
// SIG // ziV/XddaKCklKoUfP3KAlUUebeyVHv3UECQP6eGhwsrd
// SIG // d4Er4i4H3vAsAeEqYfoZtrsti4BPfsK0z2Dryo+Y9nd1
// SIG // nOPk6njd7rFeZ9IcwBIZRSZj79XdzZsNX0JBZgluMRYO
// SIG // JaJiVPdgtHFsTlXWAM6q7ItI8Kt3XIE7Vh2n4Rbh3wI/
// SIG // SP4xCLBIAY9fhqt8snMpvNfR6Xhy8yA3bJx5BT7Tj6f3
// SIG // u1PzrBpQPdcL0H013C4+gXZwI/T0+bFlvv9z54OdxjF2
// SIG // b1gT7VWzRD/Z+x9GMLZLVUT1+12o4UFU2nyD/GqVYBK1
// SIG // CcntFyKBcKGCE0owghNGBgorBgEEAYI3AwMBMYITNjCC
// SIG // EzIGCSqGSIb3DQEHAqCCEyMwghMfAgEDMQ8wDQYJYIZI
// SIG // AWUDBAIBBQAwggE9BgsqhkiG9w0BCRABBKCCASwEggEo
// SIG // MIIBJAIBAQYKKwYBBAGEWQoDATAxMA0GCWCGSAFlAwQC
// SIG // AQUABCCE/gNvYyCEJHmc7GHt2GVjNOIUZfVN9FjeZLs+
// SIG // riwdvAIGWIurMilmGBMyMDE3MDIwOTAzNDMzMS44MjRa
// SIG // MAcCAQGAAgH0oIG5pIG2MIGzMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMScwJQYDVQQLEx5uQ2lw
// SIG // aGVyIERTRSBFU046RjUyOC0zNzc3LThBNzYxJTAjBgNV
// SIG // BAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Wg
// SIG // gg7NMIIGcTCCBFmgAwIBAgIKYQmBKgAAAAAAAjANBgkq
// SIG // hkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNV
// SIG // BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
// SIG // HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEy
// SIG // MDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNh
// SIG // dGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1
// SIG // WhcNMjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFt
// SIG // cCBQQ0EgMjAxMDCCASIwDQYJKoZIhvcNAQEBBQADggEP
// SIG // ADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9plGt0VBDV
// SIG // pQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsa
// SIG // lR3OCROOfGEwWbEwRA/xYIiEVEMM1024OAizQt2TrNZz
// SIG // MFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeGMoQedGFnkV+B
// SIG // VLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIa
// SIG // IYqvS2SJUGKxXf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/Vm
// SIG // wAOWRH7v0Ev9buWayrGo8noqCjHw2k4GkbaICDXoeByw
// SIG // 6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVP
// SIG // Ik0CAwEAAaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBTVYzpcijGQ80N7fEYbxTNoWoVtVTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBTV9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBN
// SIG // MEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAx
// SIG // MC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsG
// SIG // AQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIz
// SIG // LmNydDCBoAYDVR0gAQH/BIGVMIGSMIGPBgkrBgEEAYI3
// SIG // LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6Ly93d3cubWlj
// SIG // cm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5o
// SIG // dG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8A
// SIG // UABvAGwAaQBjAHkAXwBTAHQAYQB0AGUAbQBlAG4AdAAu
// SIG // IB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+B
// SIG // cQM9naOhIW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/1
// SIG // 5/B4vceoniXj+bzta1RXCCtRgkQS+7lTjMz0YBKKdsxA
// SIG // QEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzy
// SIG // mXlKkVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8
// SIG // z+0DpZaPWSm8tv0E4XCfMkon/VWvL/625Y4zu2JfmttX
// SIG // QOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3DnKOi
// SIG // PPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKD
// SIG // uLWAmyI4ILUl5WTs9/S/fmNZJQ96LjlXdqJxqgaKD4kW
// SIG // umGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110mCIIYdqw
// SIG // UB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mB
// SIG // y6cJrDm77MbL2IK0cs0d9LiFAR6A+xuJKlQ5slvayA1V
// SIG // mXqHczsI5pgt6o3gMy4SKfXAL1QnIffIrE7aKLixqduW
// SIG // sqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8ch
// SIG // r1m1rtxEPJdQcdeh0sVV42neV8HR3jDA/czmTfsNv11P
// SIG // 6Z0eGTgvvM9YBS7vDaBQNdrvCScc1bN+NR4Iuto229Nf
// SIG // j950iEkSMIIE2jCCA8KgAwIBAgITMwAAALCG6ZIgCl3q
// SIG // +AAAAAAAsDANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
// SIG // ZS1TdGFtcCBQQ0EgMjAxMDAeFw0xNjA5MDcxNzU2NTZa
// SIG // Fw0xODA5MDcxNzU2NTZaMIGzMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMScwJQYDVQQLEx5uQ2lw
// SIG // aGVyIERTRSBFU046RjUyOC0zNzc3LThBNzYxJTAjBgNV
// SIG // BAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Uw
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDw
// SIG // 5fCNlFmpmtdsCQd3FCFZXbc9eLg1rfUwMf/O4f/W6RrJ
// SIG // g5gj+5AQwZLsOrxQbJC9XPFrrUyi9WGlh+EprKM8Et9/
// SIG // xACCzr20Cl/LuduatxktWu0HAK1U/TOs9vgSJEokZ1fa
// SIG // uEuhrA+A+Tm9IA21p8QsS/GhVubyLye5JsEzJdkrDDBy
// SIG // UIRrkmqVjPL6CE24LiTVQ9Pc6/N0aoizybRg3MllrV8J
// SIG // 5RFqFDTB5FcGEkbmoL2EWiRCQ/a89CxVmVqNs4imqhKU
// SIG // Ir6GtUqJjKpHsKDFHxuPnPBibVSdMtOpxJtT6blyO78X
// SIG // nq9YXJ3GK1Ahu9iWzDbvjaZz2a27Q3AVAgMBAAGjggEb
// SIG // MIIBFzAdBgNVHQ4EFgQU/KgHUtnvKf6YQzwVXHRet39z
// SIG // 4K8wHwYDVR0jBBgwFoAU1WM6XIoxkPNDe3xGG8UzaFqF
// SIG // bVUwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWlj
// SIG // VGltU3RhUENBXzIwMTAtMDctMDEuY3JsMFoGCCsGAQUF
// SIG // BwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNUaW1TdGFQ
// SIG // Q0FfMjAxMC0wNy0wMS5jcnQwDAYDVR0TAQH/BAIwADAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDCDANBgkqhkiG9w0BAQsF
// SIG // AAOCAQEAkv2A93W9ZZA+F83VFYPFjgKRO6xOfWDvjzkG
// SIG // k5DjD6pFPYk/Av3sb7hQkAlshNI3IZmxwYZ2HeQNxo7/
// SIG // GOCi+ka1hXd0bk4MREXQvNK2BH5wSw/WqwdpVkp2ZOj5
// SIG // qkejo4bc9M9EuEkQW2eP0dp5rjrdh1MG6I9q/H/X5KOG
// SIG // RRUNkWIiOpBK49hoAUnJLQ5reGwRAvSPTRFgc6gDIQ2X
// SIG // 4w9ydbv96A646/wgQZ2Ok/3FM3M+OXq9ajQeOUdiEbUc
// SIG // 71f0c4Nxn6gUZb7kA45NbcQBMxt+V+yh8xyXqTin9Kg6
// SIG // OfmJNfxdoyKuCr2NDKsxEm7pvWEW7PQZOiSFYl+psqGC
// SIG // A3YwggJeAgEBMIHjoYG5pIG2MIGzMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScwJQYDVQQLEx5u
// SIG // Q2lwaGVyIERTRSBFU046RjUyOC0zNzc3LThBNzYxJTAj
// SIG // BgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZp
// SIG // Y2WiJQoBATAJBgUrDgMCGgUAAxUAvIT7nVsS2sc2hTuI
// SIG // Zp6jFhjVzByggcIwgb+kgbwwgbkxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5D
// SIG // aXBoZXIgTlRTIEVTTjo1N0Y2LUMxRTAtNTU0QzErMCkG
// SIG // A1UEAxMiTWljcm9zb2Z0IFRpbWUgU291cmNlIE1hc3Rl
// SIG // ciBDbG9jazANBgkqhkiG9w0BAQUFAAIFANxGNIgwIhgP
// SIG // MjAxNzAyMDkwMDIxMjhaGA8yMDE3MDIxMDAwMjEyOFow
// SIG // dDA6BgorBgEEAYRZCgQBMSwwKjAKAgUA3EY0iAIBADAH
// SIG // AgEAAgIBxjAHAgEAAgIbMDAKAgUA3EeGCAIBADA2Bgor
// SIG // BgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMBoAowCAIB
// SIG // AAIDFuNgoQowCAIBAAIDB6EgMA0GCSqGSIb3DQEBBQUA
// SIG // A4IBAQAYjBhhG6oLiiNv6mZogqxukoDvyR7/qklj9SnQ
// SIG // gR3cq3XaBEnK5RUjqFylWQ4hdcz+VtdfPhwDB95pqsLx
// SIG // wkSuqt7PP57+qMwQxp7GzGkYz5Yww9QXfepNIlQQYkQV
// SIG // +TgoJ721V0d/lIvcg+eSIqN0BciXCIEhMfvDXqtipg31
// SIG // ftSdx4yO02s4eMTVhaIzz4PdSH0a/6iS3fIdXsPDyqjJ
// SIG // Z3ICGYK5lV3irizgVWli51yKkzN89NnIw5U1fPhscCNu
// SIG // +D8WPH+vXgq96Qe3BuyvdwiJE7cgbNafH2P8nJUoXxGb
// SIG // /l/Xn/HgGw9po4M5h/GyFY32z1CPVM+mQ3HLA+dnMYIC
// SIG // 9TCCAvECAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNV
// SIG // BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
// SIG // HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEm
// SIG // MCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENB
// SIG // IDIwMTACEzMAAACwhumSIApd6vgAAAAAALAwDQYJYIZI
// SIG // AWUDBAIBBQCgggEyMBoGCSqGSIb3DQEJAzENBgsqhkiG
// SIG // 9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQgWlGPz8upOkiF
// SIG // iQ1lsdxS1UqjJsDLC3vOvdk9FRh5yuAwgeIGCyqGSIb3
// SIG // DQEJEAIMMYHSMIHPMIHMMIGxBBS8hPudWxLaxzaFO4hm
// SIG // nqMWGNXMHDCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
// SIG // IFBDQSAyMDEwAhMzAAAAsIbpkiAKXer4AAAAAACwMBYE
// SIG // FFHUlpu9yIf7hQbEOljWLP8rKJ7cMA0GCSqGSIb3DQEB
// SIG // CwUABIIBAFlz6i/XrA8l6rpN3oIYGAUYjfDeeCORDjP4
// SIG // y8JOn7qvB2EXJscimV62QKTBsEb8kpo+k3le7y+j//ym
// SIG // zWyjGON+JxB9NZ+4b6iyAQqRXQaJ+81Sz/ouJTmBknw0
// SIG // QnIh06x/AZt98XkBvRWQfnhOp4OCuAFHwE2Vs8TDZqp5
// SIG // QTzI6onObPSaQlbYss2JjgWvNsLsjU9crTK6Y58XMJR2
// SIG // ZJGFOcb6HT2sGpIPzv+ZWdThgbu0OjLXXMJTn4MNip+y
// SIG // BE8XAA0WRyS+1dUgetP4/AwHS8zuKvzggCmfpY31lSMz
// SIG // mXbciGnHXciLse2qomVpHzWXq+eHzftua3yev/HCyjA=
// SIG // End signature block
